
(function () {
  const TARGET_FLAG = 'data-restructure-processed';
  const selectors = ['img', 'p', 'h1', 'h2', 'h3', 'button', 'a', '.banner', '.text', '.caption', '.hero', '.promo'];

  const ORDER_STACK_GAP = 24;
  const STACK_MARGIN_LEFT = 40;
  const STACK_MARGIN_TOP = 40;
  let globalClickOrder = 0;
  let stackInitialized = false;
  let stackAnchorLeft = 0;
  let stackCursorTop = 0;

  const dragHint = createDragHint();
  injectSayButton();
  injectImages();
  initializeTargets(document.body);

  const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      mutation.addedNodes.forEach((node) => {
        if (!(node instanceof HTMLElement)) return;
        if (node.matches(selectorString())) {
          prepareTarget(node);
        }
        node.querySelectorAll?.(selectorString()).forEach(prepareTarget);
      });
    });
  });

  observer.observe(document.body, { childList: true, subtree: true });

  function selectorString() {
    return selectors.join(',');
  }

  function initializeTargets(root) {
    const elements = root.querySelectorAll(selectorString());
    elements.forEach(prepareTarget);
  }

  function prepareTarget(el) {
    if (el.hasAttribute(TARGET_FLAG)) return;

    const rect = el.getBoundingClientRect();
    if (rect.height < 30 || rect.width < 100) return;

    el.setAttribute(TARGET_FLAG, 'true');
    el.classList.add('restructure-target');
    el.style.cursor = 'move';

    makeDraggable(el);
  }

  function makeDraggable(el) {
    let startDocLeft = 0;
    let startDocTop = 0;
    let startWidth = 0;
    let startHeight = 0;
    let offsetX = 0;
    let offsetY = 0;
    let dragInitialized = false;
    let startClientX = 0;
    let startClientY = 0;

    const applyPosition = (left, top) => {
      const docWidth = Math.max(document.documentElement.scrollWidth, window.innerWidth);
      const docHeight = Math.max(document.documentElement.scrollHeight, window.innerHeight);
      const maxLeft = Math.max(docWidth - startWidth, 0);
      const maxTop = Math.max(docHeight - startHeight, 0);
      const boundedLeft = Math.min(Math.max(left, 0), maxLeft);
      const boundedTop = Math.min(Math.max(top, 0), maxTop);

      el.style.left = `${boundedLeft}px`;
      el.style.top = `${boundedTop}px`;
      return { left: boundedLeft, top: boundedTop };
    };

    const initializeDragIfNeeded = (initialRect) => {
      if (dragInitialized) return;
      dragInitialized = true;
      el.dataset.restructureMoved = 'true';
      const order = assignClickOrder(el);
      detachToBody(el, initialRect);

      el.classList.add('dragging');
      el.style.width = `${startWidth}px`;
      el.style.height = `${startHeight}px`;
      el.style.position = 'absolute';
      el.style.zIndex = `${1000 + order}`;
      
      // 使用元素的当前位置，不再自动移动到堆叠位置
      // 如果元素之前有保存的位置，使用保存的位置；否则使用当前位置
      const storedLeft = el.dataset.restructureLastLeft;
      const storedTop = el.dataset.restructureLastTop;
      const useLeft = storedLeft ? Number(storedLeft) : startDocLeft;
      const useTop = storedTop ? Number(storedTop) : startDocTop;
      
      const initialPosition = applyPosition(useLeft, useTop);
      showDragHint(startWidth, startHeight, initialPosition.left, initialPosition.top);
    };

    el.addEventListener('mousedown', (e) => {
      if (e.button !== 0) return;
      if (e.target.closest('.say-box-hide')) return;
      e.preventDefault();

      const rect = el.getBoundingClientRect();
      startWidth = rect.width;
      startHeight = rect.height;
      startDocLeft = rect.left + window.scrollX;
      startDocTop = rect.top + window.scrollY;
      offsetX = e.clientX - rect.left;
      offsetY = e.clientY - rect.top;
      startClientX = e.clientX;
      startClientY = e.clientY;
      dragInitialized = false;

      const onMouseMove = (eMove) => {
        const distance = Math.hypot(eMove.clientX - startClientX, eMove.clientY - startClientY);
        if (!dragInitialized && distance > 3) {
          initializeDragIfNeeded(rect);
        }
        if (!dragInitialized) return;

        const nextLeft = eMove.pageX - offsetX;
        const nextTop = eMove.pageY - offsetY;
        const bounded = applyPosition(nextLeft, nextTop);
        updateDragHint(bounded.left, bounded.top);
      };

      const onMouseUp = (eUp) => {
        if (dragInitialized) {
          el.classList.remove('dragging');
          hideDragHint();
          // 保持鼠标释放时的位置，不再自动吸附到堆叠位置
          const finalLeft = eUp.pageX - offsetX;
          const finalTop = eUp.pageY - offsetY;
          const bounded = applyPosition(finalLeft, finalTop);
          // 保存当前位置，不再调用ensureStackPlacement
          el.dataset.restructureMoved = 'true';
          el.dataset.restructureLastLeft = String(bounded.left);
          el.dataset.restructureLastTop = String(bounded.top);
          if (!el.dataset.restructureOrder) {
            assignClickOrder(el);
          }
        }
        document.removeEventListener('mousemove', onMouseMove);
        document.removeEventListener('mouseup', onMouseUp);
      };

      document.addEventListener('mousemove', onMouseMove);
      document.addEventListener('mouseup', onMouseUp);
    });

    // 移除点击时自动吸附到堆叠位置的功能，保持元素在当前位置
    // el.addEventListener(
    //   'click',
    //   (e) => {
    //     if (el.dataset.restructureMoved === 'true') {
    //       e.preventDefault();
    //       e.stopPropagation();
    //       ensureStackPlacement({});
    //     }
    //   },
    //   true
    // );

    function ensureStackPlacement({ keepOrder = false }) {
      const rect = el.getBoundingClientRect();
      startWidth = rect.width;
      startHeight = rect.height;
      startDocLeft = rect.left + window.scrollX;
      startDocTop = rect.top + window.scrollY;

      const order = keepOrder && el.dataset.restructureOrder ? Number(el.dataset.restructureOrder) : assignClickOrder(el);

      const original = ensureOriginalDimensions(el, rect);

      const firstTimeStacked = !el.dataset.restructureStackLeft;

      detachToBody(el, rect);

      el.classList.remove('dragging');
      applySize(el, original.width, original.height);

      el.style.position = 'absolute';
      el.style.zIndex = `${1000 + order}`;
      const positioned =
        firstTimeStacked || !keepOrder
          ? reserveStackSlot(el, original.width, original.height)
          : restoreStackSlot(el, startDocLeft, startDocTop);
      applyPosition(positioned.left, positioned.top);
    }
  }

  function ensureOriginalDimensions(el, rect) {
    if (!el.dataset.restructureOriginalWidth || !el.dataset.restructureOriginalHeight) {
      let width = rect.width;
      let height = rect.height;

      if (el.tagName === 'IMG' && el instanceof HTMLImageElement) {
        const naturalWidth = el.naturalWidth || rect.width;
        const naturalHeight = el.naturalHeight || rect.height;

        if (rect.width === 0 && naturalWidth) {
          width = naturalWidth;
        }
        if (rect.height === 0 && naturalHeight) {
          height = naturalHeight;
        }

        if (naturalWidth && naturalHeight) {
          const aspectRatio = naturalHeight / naturalWidth;
          height = width * aspectRatio;
        }
      }

      el.dataset.restructureOriginalWidth = String(width);
      el.dataset.restructureOriginalHeight = String(height);
    }
    return {
      width: Number(el.dataset.restructureOriginalWidth),
      height: Number(el.dataset.restructureOriginalHeight),
    };
  }

  function applySize(el, width, height) {
    const isImage = el.tagName === 'IMG';
    const widthPx = `${Math.round(width)}px`;
    const heightPx = `${Math.round(height)}px`;

    el.style.setProperty('boxSizing', 'border-box', 'important');
    el.style.setProperty('width', widthPx, 'important');
    el.style.setProperty('maxWidth', widthPx, 'important');
    el.style.setProperty('minWidth', widthPx, 'important');

    if (isImage) {
      el.style.setProperty('height', 'auto', 'important');
      el.style.setProperty('maxHeight', heightPx, 'important');
      el.style.setProperty('minHeight', '0px', 'important');
      el.style.setProperty('objectFit', 'contain', 'important');
    } else {
      el.style.setProperty('height', heightPx, 'important');
      el.style.setProperty('maxHeight', heightPx, 'important');
      el.style.setProperty('minHeight', heightPx, 'important');
    }
  }

  function reserveStackSlot(el, width, height) {
    if (!stackInitialized) {
      stackInitialized = true;
      stackAnchorLeft = window.scrollX + STACK_MARGIN_LEFT;
      stackCursorTop = window.scrollY + STACK_MARGIN_TOP;
    }
    const assignedLeft = stackAnchorLeft;
    const assignedTop = stackCursorTop;
    stackCursorTop += height + ORDER_STACK_GAP;
    el.dataset.restructureStackLeft = String(assignedLeft);
    el.dataset.restructureStackTop = String(assignedTop);
    el.dataset.restructureStackWidth = String(width);
    el.dataset.restructureStackHeight = String(height);
    return { left: assignedLeft, top: assignedTop };
  }

  function restoreStackSlot(el, fallbackLeft, fallbackTop) {
    const storedLeft = Number(el.dataset.restructureStackLeft);
    const storedTop = Number(el.dataset.restructureStackTop);
    if (Number.isFinite(storedLeft) && Number.isFinite(storedTop)) {
      return { left: storedLeft, top: storedTop };
    }
    return { left: fallbackLeft, top: fallbackTop };
  }

  function assignClickOrder(el) {
    if (!el.dataset.restructureOrder) {
      globalClickOrder += 1;
      el.dataset.restructureOrder = String(globalClickOrder);
    }
    return Number(el.dataset.restructureOrder);
  }

  function detachToBody(el, rect) {
    if (el.dataset.restructureDetached === 'true') return;

    const computed = window.getComputedStyle(el);
    const originalMargin = computed.margin;
    const originalDisplay = computed.display;
    const preserveProps = [
      'fontFamily',
      'fontSize',
      'fontWeight',
      'fontStyle',
      'lineHeight',
      'color',
      'textTransform',
      'letterSpacing',
      'textDecoration',
      'backgroundColor',
    ];

    preserveProps.forEach((prop) => {
      el.style[prop] = computed[prop];
    });
    el.style.margin = '0';

    if (!el.__restructurePlaceholder) {
      const placeholder = document.createElement('div');
      placeholder.className = 'restructure-placeholder';
      placeholder.style.width = `${rect.width}px`;
      placeholder.style.height = `${rect.height}px`;
      placeholder.style.margin = originalMargin;
      placeholder.style.display = originalDisplay === 'inline' ? 'inline-block' : originalDisplay;
      placeholder.style.border = '1px dashed transparent';
      el.parentNode?.insertBefore(placeholder, el);
      el.__restructurePlaceholder = placeholder;
    }

    el.dataset.restructureDetached = 'true';
    document.body.appendChild(el);
  }

  function createDragHint() {
    const hint = document.createElement('div');
    hint.className = 'restructure-drag-hint';
    document.body.appendChild(hint);
    return hint;
  }

  function showDragHint(width, height, left, top) {
    dragHint.style.width = `${width}px`;
    dragHint.style.height = `${height}px`;
    dragHint.style.left = `${left}px`;
    dragHint.style.top = `${top}px`;
    dragHint.classList.add('visible');
  }

  function updateDragHint(left, top) {
    dragHint.style.left = `${left}px`;
    dragHint.style.top = `${top}px`;
  }

  function hideDragHint() {
    dragHint.classList.remove('visible');
  }

  function injectSayButton() {
    if (document.querySelector('.say-button')) return;

    const addBtn = document.createElement('button');
    addBtn.innerText = '➕ Add My Words 添加我说的话';
    addBtn.className = 'say-button';
    addBtn.addEventListener('click', () => {
      const text = prompt('你想在页面上说什么？');
      if (!text) return;
      const box = document.createElement('div');
      box.className = 'say-box';
      
      // 创建文本包装元素
      const textWrapper = document.createElement('div');
      textWrapper.className = 'say-box-text';
      textWrapper.innerText = text;
      box.appendChild(textWrapper);

      // 计算当前视口中心位置，让文本框显示在可见区域
      const viewportWidth = window.innerWidth;
      const viewportHeight = window.innerHeight;
      const scrollX = window.scrollX;
      const scrollY = window.scrollY;
      
      // 在视口中心偏上一点的位置创建，转换为页面坐标
      const baseLeft = scrollX + (viewportWidth / 2) - 150; // 150是预估文本框宽度的一半
      const baseTop = scrollY + (viewportHeight / 3); // 视口上方1/3处
      
      box.style.left = `${baseLeft}px`;
      box.style.top = `${baseTop}px`;
      box.style.position = 'absolute';
      box.style.zIndex = '10000';

      attachSayBoxControls(box);
      document.body.appendChild(box);
    });

    document.body.appendChild(addBtn);
  }

  function injectImages() {
    if (document.querySelector('.plugin-left-image') || document.querySelector('.plugin-right-image')) return;

    const leftImageUrl = chrome.runtime.getURL('image/left.png');
    const rightImageUrl = chrome.runtime.getURL('image/right.png');

    // 创建左图
    const leftImg = document.createElement('img');
    leftImg.src = leftImageUrl;
    leftImg.className = 'plugin-left-image';
    leftImg.style.position = 'fixed';
    leftImg.style.left = '10px';
    leftImg.style.top = '20%';
    leftImg.style.height = 'auto';
    leftImg.style.zIndex = '99999';
    leftImg.style.pointerEvents = 'none';
    leftImg.style.objectFit = 'contain';
    leftImg.onload = function() {
      const originalWidth = this.naturalWidth;
      this.style.width = (originalWidth * 0.2) + 'px';
    };

    // 创建右图
    const rightImg = document.createElement('img');
    rightImg.src = rightImageUrl;
    rightImg.className = 'plugin-right-image';
    rightImg.style.position = 'fixed';
    rightImg.style.right = '10px';
    rightImg.style.top = '20%';
    rightImg.style.height = 'auto';
    rightImg.style.zIndex = '99999';
    rightImg.style.pointerEvents = 'none';
    rightImg.style.objectFit = 'contain';
    rightImg.onload = function() {
      const originalWidth = this.naturalWidth;
      this.style.width = (originalWidth * 0.3) + 'px';
    };

    document.body.appendChild(leftImg);
    document.body.appendChild(rightImg);
  }

  function attachSayBoxControls(box) {
    let offsetX = 0;
    let offsetY = 0;
    let isDragging = false;

    const hideBtn = document.createElement('button');
    hideBtn.className = 'say-box-hide';
    hideBtn.innerText = '×';
    hideBtn.addEventListener('click', (e) => {
      e.stopPropagation();
      box.remove();
    });
    
    // 找到文本包装元素，如果没有则创建一个
    let textWrapper = box.querySelector('.say-box-text');
    if (!textWrapper) {
      textWrapper = document.createElement('div');
      textWrapper.className = 'say-box-text';
      textWrapper.innerText = box.innerText;
      box.innerHTML = '';
      box.appendChild(textWrapper);
    }
    textWrapper.appendChild(hideBtn);

    box.addEventListener('mousedown', (e) => {
      if (e.target === hideBtn || e.target.closest('.say-box-hide')) return;
      e.preventDefault();
      e.stopPropagation();

      const rect = box.getBoundingClientRect();
      offsetX = e.clientX - rect.left;
      offsetY = e.clientY - rect.top;
      isDragging = false;
      
      // 保存初始鼠标位置和元素尺寸
      const startClientX = e.clientX;
      const startClientY = e.clientY;
      const boxWidth = rect.width;
      const boxHeight = rect.height;

      const onMouseMove = (eMove) => {
        // 检测是否开始拖动
        if (!isDragging) {
          const distance = Math.hypot(eMove.clientX - startClientX, eMove.clientY - startClientY);
          if (distance > 3) {
            isDragging = true;
            box.style.cursor = 'grabbing';
          }
        }
        
        if (isDragging) {
          // 直接使用鼠标位置减去偏移量，保持精确位置
          const newLeft = eMove.pageX - offsetX;
          const newTop = eMove.pageY - offsetY;
          
          // 限制在文档范围内
          const docWidth = Math.max(document.documentElement.scrollWidth, window.innerWidth);
          const docHeight = Math.max(document.documentElement.scrollHeight, window.innerHeight);
          const minLeft = 0;
          const minTop = 0;
          const maxLeft = Math.max(docWidth - boxWidth, 0);
          const maxTop = Math.max(docHeight - boxHeight, 0);
          
          box.style.left = `${Math.max(minLeft, Math.min(newLeft, maxLeft))}px`;
          box.style.top = `${Math.max(minTop, Math.min(newTop, maxTop))}px`;
        }
      };

      const onMouseUp = (eUp) => {
        if (isDragging) {
          // 确保最终位置是鼠标释放的位置
          const finalLeft = eUp.pageX - offsetX;
          const finalTop = eUp.pageY - offsetY;
          
          const docWidth = Math.max(document.documentElement.scrollWidth, window.innerWidth);
          const docHeight = Math.max(document.documentElement.scrollHeight, window.innerHeight);
          const minLeft = 0;
          const minTop = 0;
          const maxLeft = Math.max(docWidth - boxWidth, 0);
          const maxTop = Math.max(docHeight - boxHeight, 0);
          
          box.style.left = `${Math.max(minLeft, Math.min(finalLeft, maxLeft))}px`;
          box.style.top = `${Math.max(minTop, Math.min(finalTop, maxTop))}px`;
          box.style.cursor = 'move';
        }
        isDragging = false;
        document.removeEventListener('mousemove', onMouseMove);
        document.removeEventListener('mouseup', onMouseUp);
      };

      document.addEventListener('mousemove', onMouseMove);
      document.addEventListener('mouseup', onMouseUp);
    });
  }
})();
